/**
 * vim: set ts=4 :
 * =============================================================================
 * SourceMod Map Config Plugin
 * Reads and loads map config cvars from file and then resets them on map end.
 *
 * SourceMod (C)2004-2007 AlliedModders LLC.  All rights reserved.
 * =============================================================================
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, version 3.0, as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * As a special exception, AlliedModders LLC gives you permission to link the
 * code of this program (as well as its derivative works) to "Half-Life 2," the
 * "Source Engine," the "SourcePawn JIT," and any Game MODs that run on software
 * by the Valve Corporation.  You must obey the GNU General Public License in
 * all respects for all other code used.  Additionally, AlliedModders LLC grants
 * this exception to all derivative works.  AlliedModders LLC defines further
 * exceptions, found in LICENSE.txt (as of this writing, version JULY-31-2007),
 * or <http://www.sourcemod.net/license.php>.
 *
 */

/*
 * Code written by Liam on 2/16/2008.
 * The purpose of this plugin was to expand upon
 * the existing map config system that the Source
 * and OrangeBox use currently.
 *
 * With the 'built-in' system, you have to be careful
 * to reset all of the cvars to defaults in servers.cfg
 * or the values will carry over after the map changes.
 *
 * This plugin keeps up with what has been loaded from
 * the cfg file and then resets them all at the end of
 * the map so that you don't have to fill your server.cfg
 * up with default values just to cause them to reset.
 *
 * If you have any questions, please feel free to reply
 * to this thread or find me in #sourcemod on irc.gamesurge.net.
 *
 */

#pragma semicolon 1
#include <sourcemod>
#include <files>

#define TYPE_INT    0
#define TYPE_FLOAT  1
#define TYPE_STRING 2

new String:g_MapName[128];
new Handle:g_CvarList;

public Plugin:myinfo = 
{
    name = "Map Configs",
    author = "Liam",
    description = "Palya konfiguracio betoltese es eltavolitasa fajlbol",
    version = "1.2",
    url = "www.wcugaming.org"
}

public OnPluginStart()
{
    RegAdminCmd("sm_reloadconfigs", Command_ReloadConfigs, ADMFLAG_RCON);
    InitializeCvars();
}

public OnPluginEnd( )
{
    if(g_CvarList != INVALID_HANDLE)
    {
        CloseHandle(g_CvarList);
        g_CvarList = INVALID_HANDLE;
    }
}

public OnMapStart()
{
    GetCurrentMap(g_MapName, sizeof(g_MapName));
}

public OnAutoConfigsBuffered( )
{
    InitializeCvars( );
    LoadMapConfigs( );
}

public OnMapEnd()
{
    new MaxSize = GetArraySize(g_CvarList);

    for(new i = 0; i < MaxSize; i++)
    {
        new Handle:cvar = GetArrayCell(g_CvarList, i);

        ResetConVar(cvar);
        CloseHandle(cvar);
        cvar = INVALID_HANDLE;
    }
    CloseHandle(g_CvarList);
    g_CvarList = INVALID_HANDLE;
}

InitializeCvars( )
{
    if(g_CvarList == INVALID_HANDLE)
        g_CvarList = CreateArray( );
    ClearArray(g_CvarList);
}
        

public Action:Command_ReloadConfigs(client, args)
{
    LoadMapConfigs( );
    return Plugin_Handled;
}

LoadMapConfigs( )
{
    decl String:filename[128];

    BuildPath(Path_SM, filename, sizeof(filename), "configs/maps/%s.cfg", g_MapName);

    if(!FileExists(filename, false))
        return;

    new Handle:file = OpenFile(filename, "r");

    if(file == INVALID_HANDLE)
    {
        LogError("LoadMapConfigs() %s fajl megnyitasa nem sikerult.", filename);
        return;
    }

    while(IsEndOfFile(file) == false)
    {
        decl String:line[256];
        decl String:arg[128], String:arg2[128];
        new len, len2;

        ReadFileLine(file, line, sizeof(line));

        if(line[0] == '\0' || line[0] == ';')
            continue;

        TrimString(line);
        len = BreakString(line, arg, sizeof(arg));

        if(len == -1)
        {
            LogError("LoadMapConfigs: %s fajl tartalmaz egy hianyos sort '%s'.", filename, arg);
            continue;
        }

        if(arg[0] == '\0' || arg[0] == '/')
            continue;

        if(arg[0] != 'C' && arg[0] != 'P')
        {
            LogError("LoadMapConfigs: %s fajl nem tartalmaz jogot ebben a sorban '%s'.", filename, arg);
            continue;
        }
        
        decl String:line2[128];
        Format(line2, sizeof(line2), "%s", line[len]);
        TrimString(line2);
        len2 = BreakString(line2, arg2, sizeof(arg2));

        if(len2 == -1)
        {
            LogError("LoadMapConfigs: %s fajl tartalmaz egy hianyos sort '%s'.", filename, arg2);
            continue;
        }

        switch(arg[0])
        {
            case 'C':
            {
                ParseCvar(arg2, line2[len2], filename);
            }

            case 'P':
            {
                ParsePluginCommand(arg2, line2[len2]);
            }
        }
    }
    CloseHandle(file);
    file = INVALID_HANDLE;
}

ParseCvar(String:cvar[], String:arg[], String:filename[])
{
    new Handle:f_Cvar = INVALID_HANDLE;

    f_Cvar = FindConVar(cvar);

    if(f_Cvar == INVALID_HANDLE)
    {
        LogError("LoadMapConfigs: Rossz cvar '%s %s' a %s fajlban.", cvar, arg, filename);
        return;
    }
    PushArrayCell(g_CvarList, f_Cvar);
    ParsePluginCommand(cvar, arg); 
}

ParsePluginCommand(String:cmd[], String:args[])
{
    ServerCommand("%s \"%s\"", cmd, args);
}